package net.vtstar.zhongtong.avi.equipment.service;

import com.alibaba.fastjson.JSONObject;
import com.github.pagehelper.PageHelper;
import lombok.extern.slf4j.Slf4j;
import net.vtstar.user.mybatis.provider.Operation;
import net.vtstar.user.mybatis.provider.OrderBy;
import net.vtstar.user.mybatis.provider.WhereFilter;
import net.vtstar.user.search.service.SearchService;
import net.vtstar.utils.asserts.ParamAssert;
import net.vtstar.zhongtong.avi.equipment.domain.EquipInfo;
import net.vtstar.zhongtong.avi.equipment.domain.PaintshopArea;
import net.vtstar.zhongtong.avi.equipment.domain.PaintshopLane;
import net.vtstar.zhongtong.avi.equipment.domain.PaintshopStation;
import net.vtstar.zhongtong.avi.equipment.mapper.BasicInfoMapper;
import net.vtstar.zhongtong.avi.equipment.mapper.PaintshopAreaMapper;
import net.vtstar.zhongtong.avi.equipment.mapper.PaintshopLaneMapper;
import net.vtstar.zhongtong.avi.equipment.mapper.PaintshopStationMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;

/**
 * @Author: xiadongwei
 * @Date: 2019/9/20 9:31
 */
@Slf4j
@Service
public class BasicInfoService {

    @Autowired
    private BasicInfoMapper basicInfoMapper;

    @Autowired
    private SearchService searchService;

    @Autowired
    private PaintshopStationMapper stationMapper;

    @Autowired
    private PaintshopAreaMapper paintshopAreaMapper;

    @Autowired
    private PaintshopLaneMapper paintshopLaneMapper;

    /**
     * 分页查询设备
     */
    public List<EquipInfo> findPageEquip(JSONObject pageFilter) {
        PageHelper.startPage(pageFilter.getIntValue("pageNum"), pageFilter.getIntValue("pageSize"));
        return basicInfoMapper.findList(searchService.pageFilter2Filter(pageFilter), EquipInfo.class);
    }

    /**
     * 新增设备
     */
    public void createEquip(EquipInfo equipInfo) {
        Assert.notNull(equipInfo, "equipInfo 不能为null！");
        basicInfoMapper.insert(fieldValidEquip(equipInfo));
    }

    /**
     * 修改设备
     */
    public void updateEquip(EquipInfo equipInfo) {
        ParamAssert.notNull(equipInfo, "equipInfo 不能为 null！");
        ParamAssert.notNull(equipInfo.getId(), "EquipInfo ID 不能为 null！");
        basicInfoMapper.update(fieldValidEquip(equipInfo));
    }

    /**
     * 根据id逻辑删除设备
     */
    public void deleteEquipById(Long id) {
        EquipInfo equipInfo = basicInfoMapper.getById(id, EquipInfo.class);
        Assert.notNull(equipInfo, "数据不存在或已删除");
        basicInfoMapper.delete(equipInfo);
    }

    /**
     * 设备新增修改操作，参数校验
     *
     * @param equipInfo
     */
    private EquipInfo fieldValidEquip(EquipInfo equipInfo) {

        // 是否更新操作
        boolean isUpdate = (null != equipInfo.getId());

        if (isUpdate) {
            EquipInfo oldData = basicInfoMapper.getById(equipInfo.getId(), EquipInfo.class);
            ParamAssert.notNull(oldData, "数据不存在或已删除");
            List<EquipInfo> equipInfos = basicInfoMapper.findList(new WhereFilter() {{
                addFilter("workshop_code", Operation.EQUAL, equipInfo.getWorkshopCode());
                addFilter("equip_type", Operation.EQUAL, equipInfo.getEquipType());
                addFilter("equip_code", Operation.EQUAL, equipInfo.getEquipCode());
            }}, EquipInfo.class);
            ParamAssert.isTrue(CollectionUtils.isEmpty(equipInfos) || !oldData.getId().equals(equipInfos.get(0).getId()),
                    "存在车间/分类/编号完全相同的设备!");
            oldData.setWorkshopCode(equipInfo.getWorkshopCode());
            oldData.setWorkshopName(equipInfo.getWorkshopName());
            oldData.setEquipType(equipInfo.getEquipType());
            oldData.setEquipCode(equipInfo.getEquipCode());
            oldData.setAlias(equipInfo.getAlias());
            oldData.setEquipFactory(equipInfo.getEquipFactory());
            oldData.setUpdateTime(new Date());
            return oldData;
        }else{
            List<EquipInfo> equipInfos = basicInfoMapper.findList(new WhereFilter() {{
                addFilter("workshop_code", Operation.EQUAL, equipInfo.getWorkshopCode());
                addFilter("equip_type", Operation.EQUAL, equipInfo.getEquipType());
                addFilter("equip_code", Operation.EQUAL, equipInfo.getEquipCode());
            }}, EquipInfo.class);
            ParamAssert.isTrue(CollectionUtils.isEmpty(equipInfos), "存在车间/分类/编号完全相同的设备!");
            equipInfo.setCreateTime(new Date());
            return equipInfo;
        }
    }

    /**
     * 分页查询工位
     */
    public List<PaintshopStation> findPageStation(JSONObject pageFilter) {
        PageHelper.startPage(pageFilter.getIntValue("pageNum"), pageFilter.getIntValue("pageSize"));
        return stationMapper.findList(searchService.pageFilter2Filter(pageFilter, new WhereFilter() {{
            addJoin(PaintshopStation.class, "area_id", PaintshopArea.class, "id");
            addJoin(PaintshopStation.class, "lane_id", PaintshopLane.class, "id");
            addOrderBy(PaintshopStation.class, "sequence", OrderBy.ASC);
        }}), PaintshopStation.class);
    }

    /**
     * 新增工位
     */
    public void createStation(PaintshopStation station) {
        Assert.notNull(station, "station 不能为null！");
        stationMapper.insert(fieldValidStation(station));
    }

    /**
     * 修改工位
     */
    public void updateStation(PaintshopStation station) {
        ParamAssert.notNull(station, "station 不能为 null！");
        ParamAssert.notNull(station.getId(), "工位 ID 不能为 null！");
        stationMapper.update(fieldValidStation(station));
    }

    /**
     * 根据id逻辑删除工位
     */
    public void deleteStationById(Long id) {
        PaintshopStation station = stationMapper.getById(id, EquipInfo.class);
        Assert.notNull(station, "数据不存在或已删除");
        stationMapper.delete(station);
    }

    /**
     * 设备新增修改操作，参数校验
     *
     * @param station
     */
    private PaintshopStation fieldValidStation(PaintshopStation station) {

        // 是否更新操作
        boolean isUpdate = (null != station.getId());

        if (isUpdate) {
            PaintshopStation oldData = stationMapper.getById(station.getId(), EquipInfo.class);
            ParamAssert.notNull(oldData, "数据不存在或已删除");
            List<PaintshopStation> stationList = stationMapper.findList(new WhereFilter() {{
                addFilter("workshop_code", Operation.EQUAL, station.getWorkshopCode());
                addFilter("area_id", Operation.EQUAL, station.getAreaId());
                addFilter("lane_id", Operation.EQUAL, station.getLaneId());
                addFilter("code", Operation.EQUAL, station.getCode());
            }}, PaintshopStation.class);
            ParamAssert.isTrue(CollectionUtils.isEmpty(stationList) || !oldData.getId().equals(stationList.get(0).getId()),
                    "存在车间/区域/车道/编号完全相同的设备!");
            oldData.setWorkshopCode(station.getWorkshopCode());
            oldData.setAreaId(station.getAreaId());
            oldData.setLaneId(station.getLaneId());
            oldData.setCode(station.getCode());
            oldData.setSequence(station.getSequence());
            oldData.setKeyStation(station.getKeyStation());
            return oldData;
        }else{
            List<PaintshopStation> equipInfos = stationMapper.findList(new WhereFilter() {{
                addFilter("workshop_code", Operation.EQUAL, station.getWorkshopCode());
                addFilter("area_id", Operation.EQUAL, station.getAreaId());
                addFilter("lane_id", Operation.EQUAL, station.getLaneId());
                addFilter("code", Operation.EQUAL, station.getCode());
            }}, PaintshopStation.class);
            ParamAssert.isTrue(CollectionUtils.isEmpty(equipInfos), "存在车间/区域/车道/编号完全相同的设备!");
            return station;
        }
    }

    public List<PaintshopArea> getPaintshopArea() {
       return paintshopAreaMapper.findList(new WhereFilter(), PaintshopArea.class);
    }

    public List<PaintshopLane> getPaintshopLane(Long areaId) {
        return paintshopLaneMapper.findList(new WhereFilter(){{
            addFilter("area_id", Operation.EQUAL, areaId);
        }}, PaintshopLane.class);
    }
}
