package net.vtstar.zhongtong.avi.equipment.controller;

import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import net.vtstar.user.search.controller.vo.SearchVo;
import net.vtstar.utils.asserts.ParamAssert;
import net.vtstar.utils.domain.Return;
import net.vtstar.zhongtong.avi.equipment.domain.EquipInfo;
import net.vtstar.zhongtong.avi.equipment.domain.PaintshopStation;
import net.vtstar.zhongtong.avi.equipment.service.BasicInfoService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;

/**
 * @Author: xiadongwei
 * @Date: 2019/9/19 18:00
 */
@Slf4j
@Api(description = "基础信息模块")
@RequestMapping("/api/basicinfo")
@RestController
public class BasicInfoController {

    @Autowired
    BasicInfoService basicInfoService;

    @ApiOperation(value = "分页查询设备列表", notes = "使用通用搜索组件，searchCode：SCADA_EQUIP_INFO\n" +
            "字段含义：\n{\n" +
            "  \"workshopName\": \"车间，字符串类型\",\n" +
            "  \"equipType\": \"设备类型，字符串类型\",\n" +
            "  \"equipCode\": \"设备编号，字符串类型\",\n" +
            "  \"alias\": \"设备名称，字符串类型\"\n" +
            "  \"equipFactory\": \"设备厂商，字符串类型\"\n" +
            "}")
    @PostMapping("/equipPage")
    public Return pageListEquip(@RequestBody SearchVo pageFilter) {
        return Return.success(new PageInfo<>(basicInfoService.findPageEquip(pageFilter.toJSONObject())));
    }

    @ApiOperation(value = "新建设备", notes = "新增参数说明，新增需传入以下参数即可：\n{\n" +
            "  \"workshopCode\": \"车间，字符串类型，长度最大20\",\n" +
            "  \"workshopName\": \"车间名称，字符串类型，长度最大50\",\n" +
            "  \"equipType\": \"设备分类，字符串类型，长度最大50\",\n" +
            "  \"equipCode\": \"设备编号，字符串类型，长度最大100\"\n" +
            "  \"alias\": \"设备名称，字符串类型，长度最大100, 非必填\"\n" +
            "  \"equipFactory\": \"设备厂商，字符串类型，长度最大100, 非必填\"\n" +
            "}")
    @PostMapping("/equipCreate")
    public Return createEquip(@Valid @RequestBody EquipInfo equipInfo) {
        basicInfoService.createEquip(equipInfo);
        return Return.success();
    }

    @ApiOperation("修改设备")
    @PutMapping("/equipUpdate")
    public Return updateEquip(@Valid @RequestBody EquipInfo equipInfo) {
        basicInfoService.updateEquip(equipInfo);
        return Return.success();
    }

    @ApiOperation("删除设备（根据id删除单个）")
    @DeleteMapping("/equipDelete/{equipId}")
    public Return deleteEquip(@PathVariable("equipId") Long equipId) {
        Assert.notNull(equipId, "Id is null");
        basicInfoService.deleteEquipById(equipId);
        return Return.success();
    }

    @ApiOperation(value = "分页查询工位列表", notes = "使用通用搜索组件，searchCode：PAINTSHOP_STATION\n" +
            "字段含义：\n{\n" +
            "  \"workshopCode\": \"车间，字符串类型\",\n" +
            "  \"areaName\": \"区域，字符串类型\",\n" +
            "  \"laneCode\": \"车道编号，字符串类型\",\n" +
            "  \"code\": \"工位编号，字符串类型\"\n" +
            "  \"keyStation\": \"是否为关键工位，boolean\"\n" +
            "}")
        @PostMapping("/stationPage")
    public Return pageListStation(@RequestBody SearchVo pageFilter) {
        return Return.success(new PageInfo<>(basicInfoService.findPageStation(pageFilter.toJSONObject())));
    }

    @ApiOperation(value = "新建工位", notes = "新增参数说明，新增需传入以下参数即可：\n{\n" +
            "  \"workshopCode\": \"车间，字符串类型，长度最大20\",\n" +
            "  \"areaId\": \"区域ID，Long，长度最大20\",\n" +
            "  \"laneId\": \"车道ID，Long，长度最大20\"\n" +
            "  \"code\": \"工位编号，字符串类型，长度最大32\"\n" +
            "  \"sequence\": \"工位序号，Int\"\n" +
            "  \"keyStation\": \"是否为关键工位，boolean\"\n" +
            "}")
    @PostMapping("/stationCreate")
    public Return createStation(@Valid @RequestBody PaintshopStation station) {
        basicInfoService.createStation(station);
        return Return.success();
    }

    @ApiOperation("修改工位")
    @PutMapping("/stationUpdate")
    public Return updateStation(@Valid @RequestBody PaintshopStation station) {
        basicInfoService.updateStation(station);
        return Return.success();
    }

    @ApiOperation("删除工位（根据id删除单个）")
    @DeleteMapping("/stationDelete/{stationId}")
    public Return deleteStation(@PathVariable("stationId") Long stationId) {
        Assert.notNull(stationId, "Id is null");
        basicInfoService.deleteStationById(stationId);
        return Return.success();
    }

    @ApiOperation(value = "查询所有区域", notes = "字段含义：\n{\n" +
            "  \"name\": \"区域名称，字符串类型\",\n" +
            "  \"onlineNum\": \"在线数量，INT\",\n" +
            "}")
    @GetMapping("/area")
    public Return getPaintshopArea() {
        return Return.success(basicInfoService.getPaintshopArea());
    }

    @ApiOperation(value = "查询指定区域下的所有车道", notes = "字段含义：\n{\n" +
            "  \"name\": \"车道名称，字符串类型\",\n" +
            "  \"code\": \"车道编号，字符串类型\",\n" +
            "}")
    @GetMapping("/lane/{areaId}")
    public Return getPaintshopLane(@PathVariable("areaId") Long areaId) {
        ParamAssert.notNull(areaId, "区域ID为NULL!");
        return Return.success(basicInfoService.getPaintshopLane(areaId));
    }
}
